/*
 * Decompiled with CFR 0.152.
 */
package org.apache.druid.indexing.worker.shuffle;

import com.google.common.base.Throwables;
import com.google.common.collect.Iterators;
import com.google.common.io.ByteSource;
import com.google.common.io.Files;
import com.google.common.util.concurrent.ListenableFuture;
import com.google.inject.Inject;
import java.io.File;
import java.io.IOException;
import java.nio.file.Path;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ThreadLocalRandom;
import java.util.concurrent.TimeUnit;
import java.util.stream.Collectors;
import java.util.stream.IntStream;
import org.apache.commons.lang3.mutable.MutableInt;
import org.apache.druid.common.guava.FutureUtils;
import org.apache.druid.common.utils.IdUtils;
import org.apache.druid.guice.ManageLifecycle;
import org.apache.druid.indexer.TaskStatus;
import org.apache.druid.indexing.common.TaskToolbox;
import org.apache.druid.indexing.common.config.TaskConfig;
import org.apache.druid.indexing.common.task.batch.parallel.GenericPartitionStat;
import org.apache.druid.indexing.worker.config.WorkerConfig;
import org.apache.druid.indexing.worker.shuffle.IntermediaryDataManager;
import org.apache.druid.java.util.common.DateTimes;
import org.apache.druid.java.util.common.FileUtils;
import org.apache.druid.java.util.common.IAE;
import org.apache.druid.java.util.common.IOE;
import org.apache.druid.java.util.common.ISE;
import org.apache.druid.java.util.common.concurrent.Execs;
import org.apache.druid.java.util.common.io.Closer;
import org.apache.druid.java.util.common.lifecycle.LifecycleStart;
import org.apache.druid.java.util.common.lifecycle.LifecycleStop;
import org.apache.druid.java.util.common.logger.Logger;
import org.apache.druid.rpc.indexing.OverlordClient;
import org.apache.druid.segment.SegmentUtils;
import org.apache.druid.segment.loading.CacheEntry;
import org.apache.druid.segment.loading.CacheEntryIdentifier;
import org.apache.druid.segment.loading.StorageLocation;
import org.apache.druid.timeline.DataSegment;
import org.apache.druid.timeline.partition.BucketNumberedShardSpec;
import org.apache.druid.utils.CompressionUtils;
import org.checkerframework.checker.nullness.qual.MonotonicNonNull;
import org.joda.time.DateTime;
import org.joda.time.Interval;
import org.joda.time.Period;
import org.joda.time.ReadablePeriod;

@ManageLifecycle
public class LocalIntermediaryDataManager
implements IntermediaryDataManager {
    private static final Logger LOG = new Logger(LocalIntermediaryDataManager.class);
    private final long intermediaryPartitionDiscoveryPeriodSec;
    private final long intermediaryPartitionCleanupPeriodSec;
    private final Period intermediaryPartitionTimeout;
    private final TaskConfig taskConfig;
    private final List<StorageLocation> shuffleDataLocations;
    private final OverlordClient overlordClient;
    private final ConcurrentHashMap<String, DateTime> supervisorTaskCheckTimes = new ConcurrentHashMap();
    private final Map<String, Iterator<StorageLocation>> locationIterators = new HashMap<String, Iterator<StorageLocation>>();
    private @MonotonicNonNull ScheduledExecutorService supervisorTaskChecker;

    @Inject
    public LocalIntermediaryDataManager(WorkerConfig workerConfig, TaskConfig taskConfig, OverlordClient overlordClient) {
        this.intermediaryPartitionDiscoveryPeriodSec = workerConfig.getIntermediaryPartitionDiscoveryPeriodSec();
        this.intermediaryPartitionCleanupPeriodSec = workerConfig.getIntermediaryPartitionCleanupPeriodSec();
        this.intermediaryPartitionTimeout = workerConfig.getIntermediaryPartitionTimeout();
        this.taskConfig = taskConfig;
        this.shuffleDataLocations = taskConfig.getShuffleDataLocations().stream().map(config -> new StorageLocation(config.getPath(), config.getMaxSize(), config.getFreeSpacePercent())).collect(Collectors.toList());
        this.overlordClient = overlordClient;
    }

    @Override
    @LifecycleStart
    public void start() {
        this.discoverSupervisorTaskPartitions();
        this.supervisorTaskChecker = Execs.scheduledSingleThreaded((String)"intermediary-data-manager-%d");
        this.supervisorTaskChecker.scheduleAtFixedRate(() -> {
            try {
                this.discoverSupervisorTaskPartitions();
            }
            catch (Exception e) {
                LOG.warn((Throwable)e, "Error while discovering supervisorTasks", new Object[0]);
            }
        }, this.intermediaryPartitionDiscoveryPeriodSec, this.intermediaryPartitionDiscoveryPeriodSec, TimeUnit.SECONDS);
        this.supervisorTaskChecker.scheduleAtFixedRate(() -> {
            try {
                this.deleteExpiredSupervisorTaskPartitionsIfNotRunning();
            }
            catch (Exception e) {
                LOG.warn((Throwable)e, "Error while cleaning up partitions for expired supervisors", new Object[0]);
            }
        }, this.intermediaryPartitionCleanupPeriodSec, this.intermediaryPartitionCleanupPeriodSec, TimeUnit.SECONDS);
    }

    @Override
    @LifecycleStop
    public void stop() {
        if (this.supervisorTaskChecker != null) {
            this.supervisorTaskChecker.shutdownNow();
            try {
                this.supervisorTaskChecker.awaitTermination(10L, TimeUnit.SECONDS);
            }
            catch (InterruptedException e) {
                Throwables.propagate((Throwable)e);
            }
        }
        this.supervisorTaskCheckTimes.clear();
    }

    private void discoverSupervisorTaskPartitions() {
        for (StorageLocation location : this.shuffleDataLocations) {
            Path locationPath = location.getPath().toPath().toAbsolutePath();
            MutableInt numDiscovered = new MutableInt(0);
            File[] dirsPerSupervisorTask = location.getPath().listFiles();
            if (dirsPerSupervisorTask != null) {
                for (File supervisorTaskDir : dirsPerSupervisorTask) {
                    String supervisorTaskId = supervisorTaskDir.getName();
                    this.supervisorTaskCheckTimes.computeIfAbsent(supervisorTaskId, k -> {
                        for (File eachFile : org.apache.commons.io.FileUtils.listFiles((File)supervisorTaskDir, null, (boolean)true)) {
                            String relativeSegmentPath = locationPath.relativize(eachFile.toPath().toAbsolutePath()).toString();
                            if (location.reserve((CacheEntry)new FileNameCacheEntry(relativeSegmentPath, eachFile.length()))) continue;
                            LOG.warn("Can't add a discovered partition[%s]", new Object[]{eachFile.getAbsolutePath()});
                        }
                        numDiscovered.increment();
                        return this.getExpiryTimeFromNow();
                    });
                }
            }
            if (numDiscovered.getValue() <= 0) continue;
            LOG.info("Discovered partitions for [%s] new supervisor tasks under location[%s]", new Object[]{numDiscovered.getValue(), location.getPath()});
        }
    }

    private void deleteExpiredSupervisorTaskPartitionsIfNotRunning() {
        HashSet<String> expiredSupervisorTasks = new HashSet<String>();
        for (Map.Entry<String, DateTime> entry : this.supervisorTaskCheckTimes.entrySet()) {
            String supervisorTaskId = entry.getKey();
            DateTime checkTime = entry.getValue();
            if (!checkTime.isBeforeNow()) continue;
            expiredSupervisorTasks.add(supervisorTaskId);
        }
        if (!expiredSupervisorTasks.isEmpty()) {
            LOG.info("Found [%s] expired supervisor tasks", new Object[]{expiredSupervisorTasks.size()});
        }
        if (!expiredSupervisorTasks.isEmpty()) {
            Map taskStatuses = (Map)FutureUtils.getUnchecked((ListenableFuture)this.overlordClient.taskStatuses(expiredSupervisorTasks), (boolean)true);
            for (Map.Entry entry : taskStatuses.entrySet()) {
                String supervisorTaskId = (String)entry.getKey();
                TaskStatus status = (TaskStatus)entry.getValue();
                if (status.getStatusCode().isComplete()) {
                    try {
                        this.deletePartitions(supervisorTaskId);
                    }
                    catch (IOException e) {
                        LOG.warn((Throwable)e, "Failed to delete partitions for task[%s]", new Object[]{supervisorTaskId});
                    }
                    continue;
                }
                this.supervisorTaskCheckTimes.put(supervisorTaskId, this.getExpiryTimeFromNow());
            }
        }
    }

    @Override
    public DataSegment addSegment(String supervisorTaskId, String subTaskId, DataSegment segment, File segmentDir) throws IOException {
        Iterator iterator = this.locationIterators.computeIfAbsent(supervisorTaskId, k -> {
            Iterator cyclicIterator = Iterators.cycle(this.shuffleDataLocations);
            int random = ThreadLocalRandom.current().nextInt(this.shuffleDataLocations.size());
            IntStream.range(0, random).forEach(i -> cyclicIterator.next());
            return cyclicIterator;
        });
        File taskTempDir = this.taskConfig.getTaskTempDir(subTaskId);
        Closer closer = Closer.create();
        closer.register(() -> {
            try {
                org.apache.commons.io.FileUtils.forceDelete((File)taskTempDir);
            }
            catch (IOException e) {
                LOG.warn((Throwable)e, "Failed to delete directory[%s]", new Object[]{taskTempDir.getAbsolutePath()});
            }
        });
        if (!(segment.getShardSpec() instanceof BucketNumberedShardSpec)) {
            throw new IAE("Invalid shardSpec type. Expected [%s] but got [%s]", new Object[]{BucketNumberedShardSpec.class.getName(), segment.getShardSpec().getClass().getName()});
        }
        BucketNumberedShardSpec bucketNumberedShardSpec = (BucketNumberedShardSpec)segment.getShardSpec();
        try (Closer resourceCloser = closer;){
            FileUtils.mkdirp((File)taskTempDir);
            File tempZippedFile = new File(taskTempDir, segment.getId().toString());
            long unzippedSizeBytes = CompressionUtils.zip((File)segmentDir, (File)tempZippedFile);
            if (unzippedSizeBytes == 0L) {
                throw new IOE("Read 0 bytes from segmentDir[%s]", new Object[]{segmentDir.getAbsolutePath()});
            }
            for (int i = 0; i < this.shuffleDataLocations.size(); ++i) {
                StorageLocation location = (StorageLocation)iterator.next();
                String partitionFilePath = this.getPartitionFilePath(supervisorTaskId, subTaskId, segment.getInterval(), bucketNumberedShardSpec.getBucketId());
                File destFile = null;
                if (!location.reserve((CacheEntry)new FileNameCacheEntry(partitionFilePath, tempZippedFile.length()))) continue;
                try {
                    destFile = new File(location.getPath(), partitionFilePath);
                    FileUtils.mkdirp((File)destFile.getParentFile());
                    FileUtils.writeAtomically((File)destFile, out -> Files.asByteSource((File)tempZippedFile).copyTo(out));
                    LOG.info("Wrote intermediary segment[%s] for subtask[%s] at [%s]", new Object[]{segment.getId(), subTaskId, destFile});
                    DataSegment dataSegment = segment.withSize(unzippedSizeBytes).withBinaryVersion(SegmentUtils.getVersionFromDir((File)segmentDir));
                    return dataSegment;
                }
                catch (Exception e) {
                    location.release((CacheEntry)new FileNameCacheEntry(partitionFilePath, tempZippedFile.length()));
                    org.apache.commons.io.FileUtils.deleteQuietly((File)destFile);
                    LOG.warn((Throwable)e, "Failed to write segment[%s] at [%s]. Trying again with the next location", new Object[]{segment.getId(), destFile});
                    continue;
                }
            }
            throw new ISE("Can't find location to handle segment[%s]", new Object[]{segment});
        }
    }

    @Override
    public Optional<ByteSource> findPartitionFile(String supervisorTaskId, String subTaskId, Interval interval, int bucketId) {
        IdUtils.validateId((String)"supervisorTaskId", (String)supervisorTaskId);
        IdUtils.validateId((String)"subTaskId", (String)subTaskId);
        for (StorageLocation location : this.shuffleDataLocations) {
            File partitionDir = new File(location.getPath(), this.getPartitionDirPath(supervisorTaskId, interval, bucketId));
            if (!partitionDir.exists()) continue;
            this.supervisorTaskCheckTimes.put(supervisorTaskId, this.getExpiryTimeFromNow());
            File segmentFile = new File(partitionDir, subTaskId);
            if (segmentFile.exists()) {
                return Optional.of(Files.asByteSource((File)segmentFile));
            }
            return Optional.empty();
        }
        return Optional.empty();
    }

    @Override
    public GenericPartitionStat generatePartitionStat(TaskToolbox toolbox, DataSegment segment) {
        return new GenericPartitionStat(toolbox.getTaskExecutorNode().getHost(), toolbox.getTaskExecutorNode().getPortToUse(), toolbox.getTaskExecutorNode().isEnableTlsPort(), segment.getInterval(), (BucketNumberedShardSpec)segment.getShardSpec(), null, null);
    }

    private DateTime getExpiryTimeFromNow() {
        return DateTimes.nowUtc().plus((ReadablePeriod)this.intermediaryPartitionTimeout);
    }

    @Override
    public void deletePartitions(String supervisorTaskId) throws IOException {
        IdUtils.validateId((String)"supervisorTaskId", (String)supervisorTaskId);
        for (StorageLocation location : this.shuffleDataLocations) {
            File supervisorTaskPath = new File(location.getPath(), supervisorTaskId);
            if (!supervisorTaskPath.exists()) continue;
            LOG.info("Cleaning up [%s]", new Object[]{supervisorTaskPath});
            for (File eachFile : org.apache.commons.io.FileUtils.listFiles((File)supervisorTaskPath, null, (boolean)true)) {
                String relativeSegmentPath = location.getPath().toPath().toAbsolutePath().relativize(eachFile.toPath().toAbsolutePath()).toString();
                location.release((CacheEntry)new FileNameCacheEntry(relativeSegmentPath, eachFile.length()));
            }
            org.apache.commons.io.FileUtils.forceDelete((File)supervisorTaskPath);
        }
        this.supervisorTaskCheckTimes.remove(supervisorTaskId);
    }

    private static class FileNameCacheEntry
    implements CacheEntry {
        private final FileNameCacheEntryIdentifier name;
        private final long size;

        private FileNameCacheEntry(String file, long size) {
            this.name = new FileNameCacheEntryIdentifier(file);
            this.size = size;
        }

        public FileNameCacheEntryIdentifier getId() {
            return this.name;
        }

        public long getSize() {
            return this.size;
        }

        public boolean isMounted() {
            return true;
        }

        public void mount(StorageLocation location) {
        }

        public void unmount() {
        }
    }

    private static final class FileNameCacheEntryIdentifier
    implements CacheEntryIdentifier {
        private final String name;

        private FileNameCacheEntryIdentifier(String name) {
            this.name = name;
        }

        public boolean equals(Object o) {
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            FileNameCacheEntryIdentifier that = (FileNameCacheEntryIdentifier)o;
            return Objects.equals(this.name, that.name);
        }

        public int hashCode() {
            return Objects.hashCode(this.name);
        }
    }
}

