/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.cdc.avro;

import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.avro.Schema;
import org.apache.avro.generic.GenericData;
import org.apache.cassandra.cdc.api.KeyspaceTypeKey;
import org.apache.cassandra.cdc.avro.AvroDataUtils;
import org.apache.cassandra.cdc.avro.msg.FieldValue;
import org.apache.cassandra.cdc.msg.CdcEvent;
import org.apache.cassandra.cdc.msg.RangeTombstone;
import org.apache.cassandra.cdc.msg.Value;
import org.apache.cassandra.cdc.schemastore.SchemaStore;
import org.apache.cassandra.spark.data.CqlField;
import org.apache.cassandra.spark.utils.MapUtils;
import org.apache.cassandra.spark.utils.Preconditions;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class CdcEventUtils {
    private static final Logger LOGGER = LoggerFactory.getLogger(CdcEventUtils.class);

    private CdcEventUtils() {
    }

    public static GenericData.EnumSymbol getAvroOperationType(CdcEvent event, Schema avroSchema) {
        return new GenericData.EnumSymbol(avroSchema.getField("operationType").schema(), (Object)CdcEventUtils.getOperationType(event));
    }

    public static OperationType getOperationType(CdcEvent event) {
        switch (event.getKind()) {
            case INSERT: {
                return OperationType.INSERT;
            }
            case UPDATE: {
                return OperationType.UPDATE;
            }
            case DELETE: 
            case ROW_DELETE: {
                return OperationType.DELETE;
            }
            case COMPLEX_ELEMENT_DELETE: {
                return OperationType.COMPLEX_ELEMENT_DELETE;
            }
            case RANGE_DELETE: {
                return OperationType.DELETE_RANGE;
            }
            case PARTITION_DELETE: {
                return OperationType.DELETE_PARTITION;
            }
        }
        throw new IllegalStateException("Unknown CDC event kind: " + String.valueOf(event.getKind()));
    }

    public static List<String> updatedFieldNames(CdcEvent event) {
        return CdcEventUtils.updatedFields(event).stream().map(value -> value.columnName).collect(Collectors.toList());
    }

    public static List<Value> updatedFields(CdcEvent event) {
        ArrayList<Value> result = new ArrayList<Value>();
        Consumer<List> addAllIfNotNull = list -> {
            if (list != null) {
                result.addAll((Collection<Value>)list);
            }
        };
        addAllIfNotNull.accept(event.getPartitionKeys());
        addAllIfNotNull.accept(event.getStaticColumns());
        addAllIfNotNull.accept(event.getClusteringKeys());
        addAllIfNotNull.accept(event.getValueColumns());
        return result;
    }

    public static List<GenericData.Record> getRangeTombstoneAvro(CdcEvent event, Schema rangeSchema, Function<Value, Object> avroFieldEncoder) {
        List<Map<String, Object>> range = CdcEventUtils.getRangeTombstone(event, avroFieldEncoder);
        if (range == null || range.isEmpty()) {
            return null;
        }
        return range.stream().map(tuple -> {
            GenericData.Record rangeRecord = new GenericData.Record(rangeSchema);
            rangeRecord.put("field", tuple.get("field"));
            rangeRecord.put("rangePredicateType", (Object)new GenericData.EnumSymbol(rangeSchema, tuple.get("rangePredicateType")));
            rangeRecord.put("value", tuple.get("value"));
            return rangeRecord;
        }).collect(Collectors.toList());
    }

    public static List<Map<String, Object>> getRangeTombstone(CdcEvent event, Function<Value, Object> encoder) {
        List rangeTombstones = event.getRangeTombstoneList();
        if (rangeTombstones == null || rangeTombstones.isEmpty()) {
            return null;
        }
        ArrayList<Map<String, Object>> result = new ArrayList<Map<String, Object>>();
        int prefix = -1;
        for (Object rt : rangeTombstones) {
            List startBounds = rt.getStartBound();
            List endBounds = rt.getEndBound();
            int p = CdcEventUtils.findLongestPrefix(startBounds, endBounds);
            if (prefix == -1) {
                prefix = p;
            }
            if (p == prefix) continue;
            LOGGER.warn("Not handling disjointed range deletion as it requires OR operator.");
            return null;
        }
        List uniqueClusteringPrefixValues = Stream.generate(() -> new HashSet()).limit(prefix).collect(Collectors.toList());
        for (RangeTombstone rt : rangeTombstones) {
            List bound = rt.getStartBound();
            for (int i = 0; i < prefix; ++i) {
                FieldValue value = new FieldValue((Value)bound.get(i));
                Set uniqeValues = (Set)uniqueClusteringPrefixValues.get(i);
                uniqeValues.add(value);
            }
        }
        for (Set valueSet : uniqueClusteringPrefixValues) {
            for (FieldValue fv : valueSet) {
                result.add(CdcEventUtils.makeRangePredicate(fv.value.columnName, valueSet.size() > 1 ? "IN" : "EQ", encoder.apply(fv.value)));
            }
        }
        RangeTombstone firstRT = (RangeTombstone)rangeTombstones.get(0);
        boolean startInclusive = firstRT.startInclusive;
        boolean endInclusive = firstRT.endInclusive;
        List start = firstRT.getStartBound();
        List end = firstRT.getEndBound();
        int longest = Math.max(start.size(), end.size());
        for (int index = prefix; index < longest; ++index) {
            Value v;
            if (index < start.size()) {
                v = (Value)start.get(index);
                result.add(CdcEventUtils.makeRangePredicate(v.columnName, startInclusive ? "GTE" : "GT", encoder.apply(v)));
            }
            if (index >= end.size()) continue;
            v = (Value)end.get(index);
            result.add(CdcEventUtils.makeRangePredicate(v.columnName, endInclusive ? "LTE" : "LT", encoder.apply(v)));
        }
        return result;
    }

    private static Map<String, Object> makeRangePredicate(String columnName, String predicateType, Object value) {
        return MapUtils.mapOf((Object[])new Object[]{"field", columnName, "rangePredicateType", predicateType, "value", value});
    }

    private static int findLongestPrefix(List<Value> bound1, List<Value> bound2) {
        int s1 = bound1.size();
        int s2 = bound2.size();
        int res = 0;
        int i = 0;
        int j = 0;
        while (i < s1 && j < s2) {
            Value v2;
            Value v1 = bound1.get(i);
            if (!CdcEventUtils.valueMatches(v1, v2 = bound2.get(j))) {
                return res;
            }
            ++i;
            ++j;
            ++res;
        }
        return res;
    }

    private static boolean valueMatches(Value v1, Value v2) {
        return Objects.equals(v1.columnName, v2.columnName) && Objects.equals(v1.columnType, v2.columnType) && Objects.equals(v1.getValue(), v2.getValue());
    }

    public static GenericData.Record getTTLAvro(CdcEvent event, Schema ttlSchema) {
        CdcEvent.TimeToLive ttl = event.getTtl();
        if (ttl == null) {
            return null;
        }
        GenericData.Record ttlRecord = new GenericData.Record(ttlSchema);
        ttlRecord.put("ttl", (Object)ttl.ttlInSec);
        ttlRecord.put("deletedAt", (Object)ttl.expirationTimeInSec);
        return ttlRecord;
    }

    public static Map<String, Integer> getTTL(CdcEvent event) {
        CdcEvent.TimeToLive ttl = event.getTtl();
        if (ttl == null) {
            return null;
        }
        return MapUtils.mapOf((Object[])new Object[]{"ttl", ttl.ttlInSec, "deletedAt", ttl.expirationTimeInSec});
    }

    public static UpdatedEvent getUpdatedEvent(CdcEvent event, SchemaStore store, int truncateThreshold, Function<KeyspaceTypeKey, CqlField.CqlType> typeLookup) {
        Schema tableSchema = store.getSchema(event.keyspace + "." + event.table, null);
        ArrayList<String> truncatedFields = new ArrayList<String>();
        GenericData.Record update = new GenericData.Record(tableSchema);
        int totalSize = 0;
        for (Value field : CdcEventUtils.updatedFields(event)) {
            ByteBuffer value = field.getValue();
            if (value == null) {
                update.put(field.columnName, null);
                continue;
            }
            if (totalSize + value.remaining() > truncateThreshold) {
                truncatedFields.add(field.columnName);
                continue;
            }
            totalSize += value.remaining();
            Schema.Field column = tableSchema.getField(field.columnName);
            Preconditions.checkNotNull((Object)column, (String)"Encountered an unknown field during event encoding. Field: %s. Avro schema: %s", (Object[])new Object[]{field.columnName, tableSchema.getFullName()});
            CqlField.CqlType type = typeLookup.apply(KeyspaceTypeKey.of((String)event.keyspace, (String)field.columnType));
            Object javaValue = type.deserializeToJavaType(value);
            update.put(field.columnName, AvroDataUtils.toAvro(javaValue, column.schema()));
        }
        return new UpdatedEvent(update, truncatedFields);
    }

    public static class UpdatedEvent {
        private final GenericData.Record record;
        private final List<String> truncatedFields;

        public UpdatedEvent(GenericData.Record record, List<String> truncatedFields) {
            this.record = record;
            this.truncatedFields = truncatedFields;
        }

        public GenericData.Record getRecord() {
            return this.record;
        }

        public List<String> getTruncatedFields() {
            return this.truncatedFields;
        }
    }

    public static enum OperationType {
        UPDATE,
        INSERT,
        DELETE,
        COMPLEX_ELEMENT_DELETE,
        DELETE_RANGE,
        DELETE_PARTITION;

    }
}

