/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.aws2.s3;

import org.apache.camel.Exchange;
import org.apache.camel.spi.Metadata;

/**
 * Constants used in Camel AWS2 S3 module
 */
public interface AWS2S3Constants {

    @Metadata(description = "The bucket Name (can be dynamic using the simple language in the producer) which this object will be stored or which will be used for the current operation or in which this object is contained.",
              javaType = "String")
    String BUCKET_NAME = "CamelAwsS3BucketName";
    @Metadata(label = "producer",
              description = "The computed bucket name that was used by the producer such as to get or put an s3 object",
              javaType = "String")
    String PRODUCED_BUCKET_NAME = "CamelAwsS3ProducedBucketName";
    @Metadata(label = "producer", description = "The bucket Destination Name which will be used for the current operation",
              javaType = "String")
    String BUCKET_DESTINATION_NAME = "CamelAwsS3BucketDestinationName";
    @Metadata(description = "Whether the bucket exists when using the headBucket operation",
              javaType = "boolean")
    String BUCKET_EXISTS = "CamelAwsS3BucketExists";
    @Metadata(description = """
            The *optional* Cache-Control HTTP header which allows the user to
            specify caching behavior along the HTTP request/reply chain.""",
              javaType = "String")
    String CACHE_CONTROL = "CamelAwsS3ContentControl";
    @Metadata(description = """
            The *optional* Content-Disposition HTTP header, which specifies
            presentational information such as the recommended filename for the
            object to be saved as.""",
              javaType = "String")
    String CONTENT_DISPOSITION = "CamelAwsS3ContentDisposition";
    @Metadata(description = """
            The *optional* Content-Encoding HTTP header specifying what content
            encodings have been applied to the object and what decoding mechanisms
            must be applied in order to obtain the media-type referenced by the
            Content-Type field.""",
              javaType = "String")
    String CONTENT_ENCODING = "CamelAwsS3ContentEncoding";
    @Metadata(description = """
            The Content-Length HTTP header indicating the size of the associated
            object in bytes.""",
              javaType = "Long")
    String CONTENT_LENGTH = "CamelAwsS3ContentLength";
    @Metadata(description = """
            The base64 encoded 128-bit MD5 digest of the associated object (content
            - not including headers) according to RFC 1864. This data is used as a
            message integrity check to verify that the data received by Amazon S3 is
            the same data that the caller sent.""",
              javaType = "String")
    String CONTENT_MD5 = "CamelAwsS3ContentMD5";
    @Metadata(description = """
            The Content-Type HTTP header, which indicates the type of content stored
            in the associated object. The value of this header is a standard MIME
            type.""",
              javaType = "String")
    String CONTENT_TYPE = "CamelAwsS3ContentType";
    @Metadata(description = """
            (producer) The ETag value for the newly uploaded object.
            (consumer) The hex encoded 128-bit MD5 digest of the associated object according to
            RFC 1864. This data is used as an integrity check to verify that the
            data received by the caller is the same data that was sent by Amazon S3.""",
              javaType = "String")
    String E_TAG = "CamelAwsS3ETag";
    @Metadata(description = "The key (can be dynamic using the simple language in the producer) under which this object is stored or will be stored or which will be used for the current operation",
              javaType = "String")
    String KEY = "CamelAwsS3Key";
    @Metadata(label = "producer", description = "The computed key that was used for retrieving this object",
              javaType = "String")
    String PRODUCED_KEY = "CamelAwsS3ProducedKey";
    @Metadata(label = "producer", description = "The Destination key which will be used for the current operation",
              javaType = "String")
    String DESTINATION_KEY = "CamelAwsS3DestinationKey";
    @Metadata(description = """
            The value of the Last-Modified header, indicating the date and time at
            which Amazon S3 last recorded a modification to the associated object.""",
              javaType = "Date")
    String LAST_MODIFIED = "CamelAwsS3LastModified";
    @Metadata(description = "The storage class of this object.", javaType = "String")
    String STORAGE_CLASS = "CamelAwsS3StorageClass";
    @Metadata(description = """
            (producer) The *optional* version ID of the newly uploaded object.
            (consumer) The version ID of the associated Amazon S3 object if available. Version
            IDs are only assigned to objects when an object is uploaded to an Amazon
            S3 bucket that has object versioning enabled.""",
              javaType = "String")
    String VERSION_ID = "CamelAwsS3VersionId";
    @Metadata(label = "producer", description = """
            The canned acl that will be applied to the object. see
            `software.amazon.awssdk.services.s3.model.ObjectCannedACL` for allowed
            values.""",
              javaType = "String")
    String CANNED_ACL = "CamelAwsS3CannedAcl";
    @Metadata(label = "producer", description = "A well constructed Amazon S3 Access Control List object.",
              javaType = "software.amazon.awssdk.services.s3.model.BucketCannedACL")
    String ACL = "CamelAwsS3Acl";
    @Metadata(description = "The operation to perform. Permitted values are copyObject, deleteObject, listBuckets, deleteBucket, listObjects",
              javaType = "String")
    String S3_OPERATION = "CamelAwsS3Operation";
    @Metadata(description = """
            Sets the server-side encryption algorithm when encrypting
            the object using AWS-managed keys. For example use AES256.""",
              javaType = "String")
    String SERVER_SIDE_ENCRYPTION = "CamelAwsS3ServerSideEncryption";
    @Metadata(label = "consumer",
              description = "If the object expiration is configured (see PUT Bucket lifecycle), the response includes this header.",
              javaType = "String")
    String EXPIRATION_TIME = "CamelAwsS3ExpirationTime";
    @Metadata(label = "consumer",
              description = "Amazon S3 can return this if your request involves a bucket that is either a source or destination in a replication rule.",
              javaType = "software.amazon.awssdk.services.s3.model.ReplicationStatus")
    String REPLICATION_STATUS = "CamelAwsS3ReplicationStatus";
    @Metadata(label = "producer", description = "The position of the first byte to get", javaType = "String")
    String RANGE_START = "CamelAwsS3RangeStart";
    @Metadata(label = "producer", description = "The position of the last byte to get", javaType = "String")
    String RANGE_END = "CamelAwsS3RangeEnd";
    @Metadata(label = "producer", description = "The expiration time of the download link in milliseconds", javaType = "Long")
    String DOWNLOAD_LINK_EXPIRATION_TIME = "CamelAwsS3DowloadLinkExpirationTime";
    @Metadata(label = "producer", description = "Whether the download link is browser compatible", javaType = "boolean")
    String DOWNLOAD_LINK_BROWSER_COMPATIBLE = "CamelAwsS3DownloadLinkBrowserCompatible";
    @Metadata(label = "producer",
              description = "The headers that are needed by the service (not needed when BrowserCompatible is true)",
              javaType = "Map<String, List<String>>")
    String DOWNLOAD_LINK_HTTP_REQUEST_HEADERS = "CamelAwsS3DownloadLinkHttpRequestHeaders";
    @Metadata(label = "producer",
              description = "The request payload that is needed by the service (not needed when BrowserCompatible is true)",
              javaType = "String")
    String DOWNLOAD_LINK_SIGNED_PAYLOAD = "CamelAwsS3DownloadLinkSignedPayload";
    @Metadata(description = """
            A map of metadata to be stored or stored with the object in S3. More details about
            metadata https://docs.aws.amazon.com/AmazonS3/latest/userguide/UsingMetadata.html[here].""",
              javaType = "Map<String, String>")
    String METADATA = "CamelAwsS3Metadata";
    @Metadata(label = "consumer", description = "The timestamp of the message", javaType = "long")
    String MESSAGE_TIMESTAMP = Exchange.MESSAGE_TIMESTAMP;

    @Metadata(description = "The prefix which is used in the com.amazonaws.services.s3.model.ListObjectsRequest to only list objects we are interested in")
    String PREFIX = "CamelAwsS3Prefix";
    @Metadata(description = "The delimiter which is used in the com.amazonaws.services.s3.model.ListObjectsRequest to only list objects we are interested in",
              javaType = "String")
    String DELIMITER = "CamelAwsS3Delimiter";
    @Metadata(description = "Return/copy the object only if its entity tag (ETag) is the same as the one specified in this header.",
              javaType = "String")
    String IF_MATCH_CONDITION = "CamelAwsS3IfMatchCondition";
    @Metadata(description = "Return/copy the object only if it has been modified since the specified time; otherwise, return a 304 Not Modified error.",
              javaType = "String")
    String IF_MODIFIED_SINCE_CONDITION = "CamelAwsS3IfModifiedSinceCondition";
    @Metadata(description = "Return/copy the object only if its entity tag (ETag) is different from the one specified in this header.",
              javaType = "String")
    String IF_NONE_MATCH_CONDITION = "CamelAwsS3IfNoneMatchCondition";
    @Metadata(description = "Return/copy the object only if it has not been modified since the specified time.",
              javaType = "String")
    String IF_UNMODIFIED_SINCE_CONDITION = "CamelAwsS3IfUnmodifiedSinceCondition";
    @Metadata(description = "The bucket Name to override (can be dynamic using the simple language in the producer) which this object will be stored or which will be used for the current operation or in which this object is contained.",
              javaType = "String")
    String OVERRIDE_BUCKET_NAME = "CamelAwsS3OverrideBucketName";
    @Metadata(label = "producer", description = "A list of keys to delete when using deleteObjects operation",
              javaType = "List<String>")
    String KEYS_TO_DELETE = "CamelAwsS3KeysToDelete";
    @Metadata(label = "producer",
              description = "The number of days for which the restore request will remain active when using restoreObject operation",
              javaType = "Integer")
    String RESTORE_DAYS = "CamelAwsS3RestoreDays";
    @Metadata(label = "producer",
              description = "The tier at which the restore will be processed when using restoreObject operation (e.g., Standard, Bulk, Expedited)",
              javaType = "String")
    String RESTORE_TIER = "CamelAwsS3RestoreTier";
    @Metadata(label = "producer", description = "The object tags to set when using putObjectTagging operation",
              javaType = "Map<String, String>")
    String OBJECT_TAGS = "CamelAwsS3ObjectTags";
    @Metadata(label = "producer", description = "The bucket tags to set when using putBucketTagging operation",
              javaType = "Map<String, String>")
    String BUCKET_TAGS = "CamelAwsS3BucketTags";
    @Metadata(label = "producer", description = "The bucket policy as JSON string when using putBucketPolicy operation",
              javaType = "String")
    String BUCKET_POLICY = "CamelAwsS3BucketPolicy";
    @Metadata(label = "producer",
              description = "The versioning status (Enabled or Suspended) when using putBucketVersioning operation",
              javaType = "String")
    String VERSIONING_STATUS = "CamelAwsS3VersioningStatus";
    @Metadata(label = "producer", description = "Whether to enable MFA delete when using putBucketVersioning operation",
              javaType = "String")
    String MFA_DELETE = "CamelAwsS3MfaDelete";
    @Metadata(label = "producer", description = "The expiration time of the upload link in milliseconds", javaType = "Long")
    String UPLOAD_LINK_EXPIRATION_TIME = "CamelAwsS3UploadLinkExpirationTime";
    @Metadata(label = "producer", description = "Whether the upload link is browser compatible", javaType = "boolean")
    String UPLOAD_LINK_BROWSER_COMPATIBLE = "CamelAwsS3UploadLinkBrowserCompatible";
    @Metadata(label = "producer",
              description = "The headers that are needed by the service for upload (not needed when BrowserCompatible is true)",
              javaType = "Map<String, List<String>>")
    String UPLOAD_LINK_HTTP_REQUEST_HEADERS = "CamelAwsS3UploadLinkHttpRequestHeaders";
    @Metadata(label = "producer",
              description = "The request payload that is needed by the service for upload (not needed when BrowserCompatible is true)",
              javaType = "String")
    String UPLOAD_LINK_SIGNED_PAYLOAD = "CamelAwsS3UploadLinkSignedPayload";

}
